<?php

namespace App\Http\Controllers\Api\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserAddress;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class ProfileController extends Controller
{
    public function getProfile(Request $request)
    {
        $user = User::where('id', Auth::user()->id)->with('userAddress')->first();

        return response()->json([
            'status' => 'ok',
            'message' => __('Registration successful'),
            'user' => $user
        ]);
    }

    public function updateProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email',
            'mobile' => 'required|numeric|min:12',
        ], [
            'name.required' => __('please_enter_name'),
            'email.required' => __('email'),
            'mobile.required' => __('number_required'),
            'mobile.numeric' => __('number_required'),
            'mobile.min' => __('number_required')
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->toArray() as $error) {
                return response()->json(['status' => 'error', 'message' => $error[0]]);
            }
        }

        $is_mobile_exist = User::where('id', '!=', Auth::user()->id)->where(['mobile' => $request->mobile])->count();
        if ($is_mobile_exist > 0) {
            return response()->json(array('status' => 'error', 'message' => __('Mobile number is already used in another account')));
        }

        $user = User::where('id', Auth::user()->id)->first();
        $user->first_name = $request->name;
        $user->email = $request->email;
        $user->mobile = $request->mobile;
        $user->save();

        $user = User::where('id', Auth::user()->id)->with('userAddress')->first();

        return response()->json([
            'status' => 'ok',
            'message' => __('Profile updated successfully'),
            'user' => $user,
        ]);
    }

    public function updateAddress(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'governorate_id' => 'required',
            'area_id' => 'required',
            'block' => 'required',
            'street' => 'required',
            // 'house_number' => 'required',
            // 'apartment' => 'required',
            // 'floor' => 'required',
            // 'other' => 'required'
        ], [
            'governorate_id.required' => __('please_enter_name'),
            'area_id.required' => __('email'),
            'block.required' => __('number_required'),
            'street.required' => __('number_required'),
            'house_number.required' => __('number_required'),
            'apartment.required' => __('password_required'),
            'floor.required' => __('password_min')
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->toArray() as $error) {
                return response()->json(['status' => 'error', 'message' => $error[0]]);
            }
        }

        $userAddress = UserAddress::where('user_id', auth()->user()->id)->first();
        $userAddress->governorate_id = $request->governorate_id ?? null;
        $userAddress->area_id = $request->area_id ?? null;
        $userAddress->block = $request->block ?? null;
        $userAddress->street = $request->street ?? null;
        $userAddress->house_number = $request->house_number ?? null;
        $userAddress->apartment = $request->apartment ?? null;
        $userAddress->floor = $request->floor ?? null;
        $userAddress->other = $request->other ?? null;
        $userAddress->save();

        $user = User::where('id', Auth::user()->id)->with('userAddress')->first();

        return response()->json([
            'status' => 'ok',
            'message' => __('Address update successful'),
            'user' => $user
        ]);
    }

    public function updatePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'old_password' => 'required|min:6',
            'new_password' => 'required|min:6',
            'new_confirm_password' => 'required|min:6',
        ], [
            'old_password.required' => __('old_password'),
            'new_password.required' => __('new_password_required'),
            'new_confirm_password.required' => __('invalid new password'),
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->toArray() as $error) {
                return response()->json(['status' => 'error', 'message' => $error[0]]);
            }
        }

        if (!Hash::check($request->old_password, auth()->user()->password)) {
            return response()->json(['status' => 'error', 'message' => __('Old Password Does not match')]);
        }

        $user = User::where('id', Auth::user()->id)->first();
        $user->password = $request->new_password;
        $user->save();

        return response()->json([
            'status' => 'ok',
            'message' => __('Password changed successfully')
        ]);
    }

    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email'
        ], [
            'email.required' => __('email')
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->toArray() as $error) {
                return response()->json(['status' => 'error', 'message' => $error[0]]);
            }
        }

        $user = User::where('type', 4)->where(['email' => $request->email])->first();
        if (!$user) {
            return response()->json(array('status' => 'error', 'message' => __('Email address does not exist')));
        }

        //send email

        return response()->json([
            'status' => 'ok',
            'message' => __('Kindly check your email for the password'),
        ]);
    }
}
