<?php

namespace App\Http\Controllers\Backend;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\View;
use App\Http\Requests\Backend\SuperadminRequest;

class SuperadminController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $totalData = User::type(3)->count();
            $limit = $request->length;
            $start = $request->start;

            $query = User::type(3)
                ->when(is_numeric($request->status), function ($query) use ($request) {
                    return $query->where('status', $request->status);
                })
                ->when($request->query('search_keyword'), function ($query) use ($request) {
                    return $query->where('first_name', 'like', '%' . $request->search_keyword . '%')
                        ->orWhere('last_name', 'like', '%' . $request->search_keyword . '%');
                })
                ->when($request->query('start_date') && $request->query('end_date'), function ($query) use ($request) {
                    return $query->whereBetween('created_at', [$request->start_date . " 00:00:00", $request->end_date . " 23:59:59"]);
                })
                ->orderBy('id', 'desc');

            $totalFiltered = $query->count();
            $items = $query->offset($start)->limit($limit)->get();

            $data = [];
            if (count($items) > 0) {
                foreach ($items as $key => $item) {
                    $nestedData['name'] = $item->first_name . ' ' . $item->middle_name . ' ' . $item->last_name;
                    $nestedData['email'] = $item->email;
                    $nestedData['mobile'] = $item->mobile ?? '-';
                    if ($item->status == 1) {
                        $nestedData['status'] = '<span class="badge badge-success p-1">Active</span>';
                    } else {
                        $nestedData['status'] = '<span class="badge badge-danger p-1">Inactive</span>';
                    }
                    $nestedData['action'] = (string)View::make('backend.superadmin.action', ['item' => $item])->render();
                    $data[$key] = $nestedData;
                }
            }

            $json_data = [
                'draw' => $request->query('draw'),
                'recordsTotal' => $totalData,
                'recordsFiltered' => $totalFiltered,
                'data' => $data
            ];
            return response()->json($json_data);
        }
        return view('backend.superadmin.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.superadmin.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required|email|unique:users,email',
            'number' => 'required|numeric|unique:users,mobile',
            'password' => 'required|min:8',
            'status' => 'required|numeric',
        ]);

        try {
            $user = new User();
            $user->type = 1;
            $user->first_name = $request->first_name;
            $user->middle_name = $request->middle_name;
            $user->last_name = $request->last_name;
            $user->mobile = $request->number;
            $user->email = $request->email;
            $user->password = $request->password;
            $user->created_by = auth()->id();
            $user->status = $request->status;
            $user->save();

            return redirect()->route('backend.super-admin.index')->with('success', 'Data created successfully.');
        } catch (Exception $e) {
            return redirect()->back()->with('danger', $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = User::find($id);
        return view('backend.superadmin.updatePassword', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = User::find($id);
        return view('backend.superadmin.update', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required|email|unique:users,email,' . $_REQUEST['user_id'],
            'number' => 'required|numeric|unique:users,mobile,' . $_REQUEST['user_id'],
            'status' => 'required|numeric',
        ]);

        try {
            $user = User::findOrFail($id);
            $user->first_name = $request->first_name;
            $user->middle_name = $request->middle_name;
            $user->last_name = $request->last_name;
            $user->mobile = $request->number;
            $user->email = $request->email;
            $user->status = $request->status;
            $user->save();

            return redirect()->route('backend.super-admin.index')->with('success', 'Data updated successfully.');
        } catch (Exception $e) {
            return redirect()->back()->with('danger', $e->getMessage());
        }
    }
    public function update_password(Request $request)
    {
        $request->validate([
            'new_password' => 'required|min:6',
            'confirm_password' => 'required_with:new_password|same:new_password|min:6',
        ]);

        try {
            $user = User::where('id',$request->user_id)->first();
            $user->password = $request->new_password;
            $user->save();

            return redirect()->route('backend.super-admin.index')->with('success', 'Data updated successfully.');
        } catch (Exception $e) {
            return redirect()->back()->with('danger', $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        User::find($id)->delete();
        return redirect()->back()->with('success', 'Item(s) deleted successfully.');
    }
}
